/**********************************************************************************************
* Device      : AD18F08
* Author      : fengxuanmian
* Version     : V0.0.1
* Date        : 2022.3.28
* Description : 配置tmr0输出四路PWM 频率为100hz 占空比为25% PA0 PA4正向输出 PA2 PB3反向输出
***********************************************************************************************/
#include "AD18F08.h"
/*********************************************************************************************
* 常用配置字功能说明：
*
* _FINTOSC_DIV1     :  内部时钟分频选择
* _DBG_OFF          :  禁用DEBUG模式（使能后DEBUG口不能做普通IO用）
* _LVREN_OFF         :  LVR使能位
* _LVR24V           :  选择LVR电压为2.4V
* _FLASH_CP1_OFF    :  1K~2K*16 FLASH保护位
* _FLASH_CP0_OFF    :  0~1K*16 FLASH保护位
* _IDIS_OFF	     	:  模拟IO打开，数据读取控制位
* _CKSUM_OFF        :  使能程序校验
* _CBP_OFF			:  代码保护选择位
* _RESETE_OFF       :  禁用外部复位
* _RSEL				:  端口上下拉电阻选择
* _STV_ON		    :  堆栈溢出复位使能位
* _PSUT_2		    :  上电复位时间选择位
* _FCPU_4T          :  选择指令周期为4T模式
* _WDTE_OFF         :  看门狗使能位
* _TWDT_000         :  看门狗溢出时间选择位
* _OSCM_HIRC_16M    :  选择内部高速时钟16MHz
* _ADCSP_ON			:  ADC采集速度选择位
* 注意：1.配置字若无特别需求请勿修改；
*       2.其他配置字及相应功能见头文件；
************************************************************************************************/
__CONFIG(1,_FINTOSC_DIV1_1L     & _DBG_OFF_1L       & _LVREN_OFF_1L     & _LVR24V_1L                                                  );
__CONFIG(2,_FLASH_CP1_OFF_2L    & _FLASH_CP0_OFF_2L & _IDIS_OFF_2L     & _CKSUM_OFF_2L  & _CBP_OFF_2L   & _RESETE_OFF_2L & _RSEL_LOW_2L & _STV_ON_2L  );
__CONFIG(3,_PSUT_2_3L           & _FCPU_4T_3L       & _WDTE_OFF_3L    & _TWDT_100_3L                                                );
__CONFIG(4,_OSCM_HIRC_16M_4L    & _MTPDIS_OFF_4L    &_ADCSP_ON_4L                                                                                      );

/**********************************************TIMER*****************************************/
#define           T0_CLK_EN            1    //使能TIMER0
#define           T0_CLK_DIS           0    //禁用TIMER0

#define           T0_FOSC_2            0    //TIMER0预分频为2
#define           T0_FOSC_4            1    //TIMER0预分频为4
#define           T0_FOSC_8            2    //TIMER0预分频为8
#define           T0_FOSC_16           3    //TIMER0预分频为16
#define           T0_FOSC_32           4    //TIMER0预分频为32
#define           T0_FOSC_64           5    //TIMER0预分频为64
#define           T0_FOSC_128          6    //TIMER0预分频为128
#define           T0_FOSC_256          7    //TIMER0预分频为256

#define           T0_CS_CPU            0    //TIMER0时钟源为CPU 运行时钟
#define           T0_CS_T0CK1          1    //TIMER0时钟源为T0CKI(PA5)
#define           T0_CS_2K             2    //TIMER0时钟源为内部2K
#define           T0_CS_CMPOUT         3    //TIMER0时钟源是CMPOUT输出

#define           T0_EN_NFD            1    //使能TIMER0的时钟且不分频
#define           T0_EN_FD             0    //使能TIMER0的时钟源分频

#define           T0_INT_EN            1    //使能TIMER0中断
#define           T0_INT_DIS           0    //禁止TIMER0中断

#define           T0_SE_U           0    //T0CKI脚上升沿触发计数
#define           T0_SE_D           1    //T0CKI脚下降沿触发计数


/**********************************************TMR0_PWM0*****************************************/

#define              ON               1      //使能PWM输出
#define              OFF              0      //禁止PWM输出

#define              OUT_N            1      //PWM正向输出
#define              OUT_P            0      //PWM反向输出

/**********************************************TMR0_PWM0*****************************************/


void GPIO_Init(void);
void Timer0_Timer(unsigned char T0_CS,unsigned char T0_SE,unsigned char CS_EN,unsigned char T0_FD);
void TMR0_PWM0(unsigned char PWM0EN,unsigned char PWM1EN,unsigned char PWM2EN,unsigned char PWM3EN,unsigned char POLS0,unsigned char POLS1,unsigned char POLS2,unsigned char POLS3);
//
//         时钟频率/计数分频/指令周期 = 计数频率           (指令周期默认配置字为4T)
//              16M/256/4  =  15.625K
//     一次计数时间:1s/15625 = 0.064ms
//需要定时10ms  需要计数 10ms/0.064ms = 156.25    故T0周期取156

void main (void)
{

	GPIO_Init();     //所有IO初始化
    //             时钟源选择CPU(16M)  触发方式（下沿）   T0使能分频   256分频
	Timer0_Timer(  T0_CS_CPU,         T0_SE_D,          T0_EN_FD,    T0_FOSC_256);        //只有使用T0CKI作为时钟源时才需要考虑配置触发方式

	PWM0DUTY = 0X27;    //配置PWM0的占空比  占空比为PWM0DUTY/T0  PWM频率与T0中断频率一致
    //           PA0 PA2 PA4 PB3  PA0正向  PA2反向  PA4正向   PB3反向
	TMR0_PWM0(ON, ON, ON, ON,  OUT_P,   OUT_N,   OUT_P,   OUT_N);    //ON-使能io输出pwm    OFF-禁止io输出pwm  OUT_P-正向输出 OUT_N-反向输出
	TMR0IF = 0;
	TMR0IP = 1;
	TMR0IE = 1;

	IPEN = 1;	   //使能中断优先级
	GIEH = 1;      //使能高优先级全局中断
	while(1)
	{

	}
}

//高优先级中断服务函数

void interrupt ISR_h (void)
{
//进入中断后系统自动关闭中断
	if((TMR0IE == 1)&&(TMR0IF == 1))
	{                                    //中断定时：时钟频率/计数分频/指令周期 = 计数频率
		TMR0IF = 0;                      //          16M/256/4  =  15.625K
		TMR0 = 0X00;                     //     一次计数时间:1s/15625 = 0.064ms
	}
}

//低优先级中断服务函数
void interrupt low_priority ISR_l (void)
{
	//该范例中未使用
}

/*******************************************************************************
// ----Function: GPIO_Init
// --Parameters: None
// -----Returns: None
//---------Note:所有IO初始化（关闭所有IO上下拉、开漏、模拟通道；所有IO设置为输出并输出0）
*******************************************************************************/
void GPIO_Init(void)
{
	TRISA = 0X00;   //PA输出模式
	TRISB = 0X00;   //PB输出模式

	PORTA = 0X00;	//PORTA输出0
	PORTB = 0X00;	//PORTB输出0

	PAPD = 0XFF;    //关闭PA下拉
	PAPU = 0XFF;    //关闭PA上拉
	PBPD = 0XFF;    //关闭PB下拉
	PBPU = 0XFF;    //关闭PB上拉

	PAOD = 0X00;    //关PA开漏输出
	PBOD = 0X00;    //关PB开漏输出

	ANASEL0 = 0X00; //关PA模拟量通道
	ANASEL1 = 0X00; //关PB模拟量通道
}
/*******************************************************************************
*  Function: timer0 定时
*  Parameters：
*
*
*  T0_CS  （时钟源）：
*                       T0_CS_CPU            0    //TIMER0时钟源为CPU 运行时钟
*                       T0_CS_T0CK1          1    //TIMER0时钟源为T0CKI(PA5)
*                       T0_CS_2K             2    //TIMER0时钟源为内部2K
*                       T0_CS_CMPOUT         3    //TIMER0时钟源是CMPOUT输出
*
*  T0_SE （触发方式）：
*                       T0_SE_U             0     //T0CKI脚上升沿触发计数       (只有使用T0CKI作为时钟源时才需要考虑配置触发方式)
*                       T0_SE_D             1     //T0CKI脚下降沿触发计数
*
*  CS_EN （分频使能）：
*                       T0_EN_FD             0    //使能TIMER0的时钟源分频
*                       T0_EN_NFD            1    //使能TIMER0的时钟且不分频
*
*  FD   （分频数）：
*                       T0_FOSC_2            0    //TIMER0预分频为2
*                       T0_FOSC_4            1    //TIMER0预分频为4
*                       T0_FOSC_8            2    //TIMER0预分频为8
*                       T0_FOSC_16           3    //TIMER0预分频为16
*                       T0_FOSC_32           4    //TIMER0预分频为32
*                       T0_FOSC_64           5    //TIMER0预分频为64
*                       T0_FOSC_128          6    //TIMER0预分频为128
*                       T0_FOSC_256          7    //TIMER0预分频为256
*
*******************************************************************************/
/*******************************************************************************
// ----Function: Timer0_Timer
// --Parameters: None
// -----Returns: None
//---------Note: 设置tmr0的配置
*******************************************************************************/
//                T0时钟源选择         T0触发方式                   T0使能分频          T0 时钟分频数
void Timer0_Timer(unsigned char T0_CS,unsigned char T0_SE,unsigned char CS_EN,unsigned char T0_FD)
{

	if (T0_CS == T0_CS_T0CK1)
	{
		TRISA5 = 1;            //PA5设置输入
	}
	T0CON = 0X00;           //
	T0CON |= T0_CS << 5;    //时钟源选择
	T0CON |= T0_SE << 4;    //T0CKI计数触发方式
	T0CON |= CS_EN << 3;    //分频使能
	T0CON |= T0_FD;         //分频数
	PR0 = 156;              //计数周期
	TMR0 = 0;               //计数初值
	T0CLKEN = 1;            //使能T0的时钟

}
/*******************************************************************************
// ----Function: TMR0_PWM0
// --Parameters: None
// -----Returns: None
//---------Note:设置PWM1/2/3/4的使能位和极性
*******************************************************************************/
void TMR0_PWM0(unsigned char PWM0EN,unsigned char PWM1EN,unsigned char PWM2EN,unsigned char PWM3EN,unsigned char POLS0,unsigned char POLS1,unsigned char POLS2,unsigned char POLS3)
{

	PWM0CON = 0X00;           //
	PWM0CON |= POLS3   << 7;  //PWM3极性选择
	PWM0CON |= POLS2   << 6;  //PWM2极性选择
	PWM0CON |= POLS1   << 5;  //PWM1极性选择
	PWM0CON |= POLS0   << 4;  //PWM0极性选择
	PWM0CON |= PWM3EN << 3;   //PWM3(PB3)使能位
	PWM0CON |= PWM2EN << 2;   //PWM2(PA4)使能位
	PWM0CON |= PWM1EN << 1;   //PWM1(PA2)使能位
	PWM0CON |= PWM0EN ;       //PWM0(PA0)使能位

}
